{************************************************************************}
{************************************************************************}
{* Modul:       TIFFChk_Frm.pas                                         *}
{************************************************************************}
{* Inhalt:      Hauptfenster mit Auswahl und Analyse von TIFF-Dateien   *}
{************************************************************************}
{* Funktion:    Ausgabe der Datei-Informationen (Pfad/Name/Gre/Flags),*}
{*              Analyse der Tag-Struktur und Ausgabe aller Tags zusammen*}
{*              mit den wichtigsten Inhalten in einem String-Grid       *}  
{************************************************************************}
{* Version:     0.1.0 B002                                              *}
{* Autor:       Thomas Mainka                                           *}
{* Datum:       28.Jun.2016                                             *}
{* Vernderung: Ausgabe fr Multiwert-Tags (z.B. $0129) erweitert       *}
{*              Auswertung und Anzeige auch von Folge-IFDs (MultiImage) *}
{*              Datenausgabe um div Tags erweitert                      *}
{************************************************************************}
{* Revision:  V0.1.0 B001 (24.Jun.2016):                                *}
{*              Drag&Drop-Interface vom Explorer eingebaut              *}
{*              Kommentar-Header hinzugefgt                            *}
{*              Korrektur Formular-berschrift                          *}
{*              Datenausgabe um div. Tags erweitert                     *}
{*            V0.1.0 B000:                                              *}
{*              Erste Version im SVN-Repository zum Aufbau des SK       *}
{************************************************************************}
{* Class:       TForm1                                                  *}
{* Methoden:    TForm1.GetFileInfo                                      *}
{*              TForm1.GetTIFFInfo                                      *}
{*              TForm1.GetEXIFInfo(z.Z. leer)                           *}
{*              TForm1.WMDROPFILES                                      *}
{* HRoutinen:   LongW                                                   *}
{************************************************************************}

unit TIFFChk_Frm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Menus, StdCtrls, Grids, Math, ShellApi;

type
  Byte4     = Array[1..4] of Byte;
  TifType   = packed record
                case Byte of
                  0: (Tag: Word;
                      Typ: Word;
                      WDum: LongInt;
                      WOf1: Word;
                      WOf2: Word;);
                  1: (IFH: LongInt;
                      ICnt: LongInt;
                      IOfs: LongWord;);
                  2: (MFH: Byte4;
                      MCnt: Byte4;
                      MOfs: Byte4;);
                  3: (BTag: Word;
                      BTyp: Word;
                      BDum: Longint;
                      BOf1: Byte;
                      BOf2: Byte;
                      BOf3: Byte;
                      BOf4: Byte;);
              end;
  TBufType  = packed record
                case Byte of
                  0: (IW: Array[0..24] of Longint;);
                  1: (MW: Array[0..24] of Byte4;);
                  2: (WW: Array[0..49] of Word;);
                  3: (BW: Array[0..99] of Byte;);
                  4: (CW: Array[0..99] of AnsiChar;);
              end;
  FBufType  = packed record
                case Integer of
                  -1: (DummyP: Array[0..4095] of AnsiChar;);
                   0: (DummyA: Array[0..4095] of Byte;);
                   1: (JFIFKen: Word;
                       JFIFLen: Word;
                       DummyB: Array[0..4091] of AnsiChar;);
                   2: (FFE0Ken: Word;
                       FFE0Len: Word;
                       FFE0Ide: Array[0..4] of AnsiChar;
                       FFE0Ver: Word;
                       FFE0AEi: Byte;
                       FFE0HAu: Word;
                       FFE0VAu: Word;
                       FFE0VHo: Byte;
                       FFE0VVe: Byte;
                       FFE0Dum: Array[14..4091] of AnsiChar;);
                   3: (FFC0Ken: Word;
                       FFC0Len: Word;
                       FFC0Pre: Byte;
                       FFC0Hor: Word;
                       FFC0Ver: Word;
                       FFC0CCp: Byte;
                       FFC0Id1: Byte;
                       FFC0SF1: Byte;
                       FFC0QT1: Byte;
                       FFC0Id2: Byte;
                       FFC0SF2: Byte;
                       FFC0QT2: Byte;
                       FFC0Id3: Byte;
                       FFC0SF3: Byte;
                       FFC0QT3: Byte;
                       FFC0Dum: Array[15..4091] of AnsiChar;);
                   4: (FFC4Ken: Word;
                       FFC4Len: Word;
                       FFC4CCp: Byte;
                       FFC4HTL: Array[0..15] of Byte;
                       FFC4HTa: Array[17..4091] of Byte;);
                   5: (FFCCKen: Word;
                       FFCCLen: Word;
                       FFCCCCp: Byte;
                       FFCCWAC: Byte;
                       FFCCDum: Array[2..4091] of AnsiChar;);
                   6: (FFDDKen: Word;
                       FFDDLen: Word;
                       FFDDInt: Word;
                       FFDDDum: Array[2..4091] of AnsiChar;);
                  20: (TifCnt: Word;
                       TifTag: Array[1..42] of TifType;
                       TifDum: Array[0..3583] of Byte;);
             end;
  TForm1 = class(TForm)
    MainMenu1: TMainMenu;
    OpenDialog1: TOpenDialog;
    StringGrid1: TStringGrid;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    Label8: TLabel;
    Label9: TLabel;
    Label10: TLabel;
    Datei1: TMenuItem;
    Open1: TMenuItem;
    Close1: TMenuItem;
    Label11: TLabel;
    Label12: TLabel;
    procedure Open1Click(Sender: TObject);
    procedure Close1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    { Private-Deklarationen }
    Line: Integer;
    FBuf: FBufType;
    FLen: int64;
    procedure GetFileInfo(FileName:string);
    procedure GetTIFFInfo(FH:integer);
    procedure GetEXIFInfo(FH:integer);
    procedure WMDROPFILES(var Msg: TMessage); message WM_DROPFILES;  public
  public
    { Public-Deklarationen }
  end;

var
  Form1: TForm1;

implementation

{$R *.dfm}

{************************************************************************}
{* Routine:     LongW                                       (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Umwandlung eines Byte4-Langwortes in ein Langwort       *}
{* Parameter:   x:Byte4                                                 *}
{*               = 4-Byte Array in Big-Endian Folge                     *}
{* Rckgabe:    Result:LongWord                                         *}
{*               = LongWord (little-Endian)                             *}
{************************************************************************}

Function LongW(x:Byte4):LongWord;
begin
   LongW:=x[4]+Word(x[3])*256+x[2]*65536+x[1]*16777216;
end;

{************************************************************************}
{* Pr.-Methode: GetFileInfo                                 (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Ermitteln und Anzeige der Datei-Eigenschaften, ffnen   *}
{*              der Datei und Aufruf der Typ-Spezifischen Analyse-      *}
{*              Routine (hier GetTIFFInfo)                              *}
{* Parameter:   FileName:string                                         *}
{*               = kompletter Dateiname (incl Pfad)                     *}
{************************************************************************}

procedure TForm1.GetFileInfo(FileName:string);
Var FH   : integer;
    Addr : int64;
    Attr : integer;
    AStr : Array[0..7] of AnsiChar;
    HStr : Array[0..255] of AnsiChar;
begin
  Label2.Caption:=ExtractFilePath(FileName);
  Label4.Caption:=ExtractFileName(FileName);
  Attr:=FileGetAttr(FileName);
  AStr:='--------';
  if (Attr and faReadOnly) > 0 then AStr[7]:='R';
  if (Attr and faHidden) > 0   then AStr[6]:='H';
  if (Attr and faSysFile) > 0  then AStr[5]:='S';
  if (Attr and faArchive) > 0  then AStr[3]:='A';
  if (Attr and faSymLink) > 0  then AStr[2]:='J';
  Label8.Caption:=AStr;
  Label10.Caption:=DateTimeToStr(FileDateToDateTime(FileAge(FileName)));
  StrPCopy(HStr,FileName);
  FH:=FileOpen(HStr,of_Read+of_Share_Compat);
  if (FH=-1) then begin
    FH:=FileOpen(HStr,of_Read+of_Share_Deny_none);
  end;
  if FH>0 then begin
    FLen:=FileSeek(FH,0,2);
    Label6.Caption:=InttoStr(FLen)+' Bytes';
    Addr:=FileSeek(FH,0,0);
    Addr:=Addr+FileRead(FH,FBuf.DummyP,8);
    GetTIFFInfo(FH);
    FileClose(FH);
  end
  else begin
    Label12.Caption:='File could not opened';
  end;
end;

{************************************************************************}
{* Pr.-Methode: GetTIFFInfo                                 (0.1.0 B002)*}
{************************************************************************}
{* Inhalt:      Analyse und Ausgabe der TIFF-Tags im StringGrid         *}
{* Parameter:   FH: Integer                                             *}
{*               = File-Handle (Windows-Funktionen)                     *}
{************************************************************************}

procedure TForm1.GetTIFFInfo(FH: Integer);
Var Adr : Int64;
    Len : integer;
    Mot : Boolean;
    i   : integer;
    HStr: AnsiString;
    cntP: integer;
    cntD: integer;

  {**********************************************************************}
  {* Routine:     SwapTIFF_IFD                              (0.1.0 B000)*}
  {**********************************************************************}
  {* Inhalt:      Umwandlung der Werte der TIFF-IFD-Struktur von        *}
  {*              Big-Endian nach Little-Endian                         *}
  {**********************************************************************}

  procedure SwapTIFF_IFD;
  Var i: Integer;
  begin
    FBuf.TIFCnt:=Swap(FBuf.TIFCnt);
    for i:=1 to FBuf.TifCnt do begin
      FBuf.TIFTag[i].Tag:=Swap(FBuf.TIFTag[i].Tag);
      FBuf.TIFTag[i].Typ:=Swap(FBuf.TIFTag[i].Typ);
      FBuf.TIFTag[i].ICnt:=LongW(FBuf.TIFTag[i].MCnt);
      if (FBuf.TIFTag[i].ICnt>1) or (FBuf.TIFTag[i].Typ=4) or
         (FBuf.TIFTag[i].Typ=9) or (FBuf.TIFTag[i].Typ=11) then
        FBuf.TIFTag[i].IOfs:=LongW(FBuf.TIFTag[i].MOfs);
      if (FBuf.TIFTag[i].ICnt=1) and
         ((FBuf.TIFTag[i].Typ=3) or (FBuf.TIFTag[i].Typ=8)) then
        FBuf.TIFTag[i].WOf1:=Swap(FBuf.TIFTag[i].WOf1);
    end;
    FBuf.TIFTAG[FBuf.TifCnt+1].IFH:=LongW(FBuf.TIFTAG[FBuf.TifCnt+1].MFH);
  end;

  {**********************************************************************}
  {* Routine:     GetTiffStr                                (0.1.0 B002)*}
  {**********************************************************************}
  {* Inhalt:      Ermittlung eines (oder bis zu anz Werten) aus der akt.*}
  {*              IFD (bzw. dem zug. Buffer-Bereichs) mit ggf. notw.    *}
  {*              Big-Endian/Little-Endian Konvertierung als String     *}
  {* Parameter    i: Integer                                            *}
  {*               = Element innerhalb der IFD-Struktur                 *}
  {*              anz: Integer (Default 1)                              *}
  {*               = Anzahl der Auszugebenen Werte                      *}
  {*              cTr: AnsiChar (Default '/')                           *}
  {*               = Trennzeichen zwischen den Werten                   *}
  {* Rckgabe:    Result:Ansistring                                     *}
  {*               = Ausgabe vom Wert/String/Wertauflistung als String  *}
  {**********************************************************************}

  function GetTiffStr(i: Integer; anz: Integer=1; cTr: AnsiChar='/'):AnsiString;
  Var Ext : Boolean;
      xPos: int64;
      xLen: Int64;
      TBuf: TBufType;
      xi  : Integer;
      xr  : Double;
  begin
    Ext:=True;
    Result:='';
    if FBuf.TifTag[i].ICnt=1 then begin
      case FBuf.TIFTag[i].Typ of
        1 : begin
              Result:=IntToStr(Word(FBuf.TIFTag[i].BOf1));
              Ext:=False;
            end;
        3 : begin
              Result:=IntToStr(Word(FBuf.TIFTag[i].WOf1));
              Ext:=False;
            end;
        4 : begin
              Result:=IntToStr(FBuf.TIFTag[i].IOfs);
              Ext:=False;
            end;
      end;
    end;
    if Ext and (FBuf.TifTag[i].ICnt=2) and (anz=2) then begin
      case FBuf.TIFTag[i].Typ of
        3 : begin
              if Mot then
                Result:=IntToStr(Word(FBuf.TIFTag[i].WOf2))+cTr+
                        IntToStr(Word(FBuf.TIFTag[i].WOf1))
              else
                Result:=IntToStr(Word(FBuf.TIFTag[i].WOf1))+cTr+
                        IntToStr(Word(FBuf.TIFTag[i].WOf2));
              Ext:=False;
            end;
      end;
    end;
    if Ext and (FBuf.TIFTag[i].Typ=2) then begin
      xPos:=FileSeek(FH,FBuf.TIFTag[i].IOfs,0);
      xLen:=FileRead(FH,TBuf,min(FBuf.TifTag[i].ICnt,100));
      xi:=0;
      while (xi<xLen) and (TBuf.BW[xi]<>$00) do begin
        Result:=Result+TBuf.CW[xi];
        inc(xi);
      end;
    end;
    if Ext and (FBuf.TIFTag[i].Typ=3) then begin
      xPos:=FileSeek(FH,FBuf.TIFTag[i].IOfs,0);
      xLen:=FileRead(FH,TBuf,min(FBuf.TifTag[i].ICnt*2,100));
      xi:=0;
      while (xi<xLen/2) and (xi<anz) do begin
        if xi>0 then
          Result:=Result+' '+cTr+' ';
        if Mot then
          Result:=Result+IntToStr(Swap(TBuf.WW[xi]))
        else
          Result:=Result+IntToStr(TBuf.WW[xi]);
        inc(xi);
      end;
    end;
    if Ext and (FBuf.TIFTag[i].Typ=4) then begin
      xPos:=FileSeek(FH,FBuf.TIFTag[i].IOfs,0);
      xLen:=FileRead(FH,TBuf,min(FBuf.TifTag[i].ICnt*4,100));
      xi:=0;
      Result:='';
      while (xi<xLen/4) and (xi<anz) do begin
        if xi>0 then
          Result:=Result+' '+cTr+' ';
        if Mot then
          Result:=Result+IntToStr(LongW(TBuf.MW[xi]))
        else
          Result:=Result+IntToStr(TBuf.IW[xi]);
        inc(xi);
      end;
    end;
    if Ext and (FBuf.TIFTag[i].Typ=5) then begin
      xPos:=FileSeek(FH,FBuf.TIFTag[i].IOfs,0);
      xLen:=FileRead(FH,TBuf,min(FBuf.TifTag[i].ICnt*8,100));
      xi:=0;
      Result:='';
      while (xi<xLen/8) and (xi<anz) do begin
        if xi>0 then
          Result:=Result+' '+cTr+' ';
        if Mot then
          xr:=LongW(TBuf.MW[xi*2])/LongW(TBuf.MW[xi*2+1])
        else
          xr:=TBuf.IW[xi*2]/TBuf.IW[xi*2+1];
        Result:=Result+FloatToStr(xr);
        inc(xi);
      end;
    end;
  end;

  {**********************************************************************}
  {* Routine:     GetTIFFVal_I                              (0.1.0 B002)*}
  {**********************************************************************}
  {* Inhalt:      Ermittlung eines Integer-Wertes aus der akt. IFD      *}
  {*              (bzw. dem zug. Buffer-Bereichs) mit ggf. notw. Big-   *}
  {               Endian/Little-Endian Konvertierung                    *}
  {* Parameter    i: Integer                                            *}
  {*               = Element innerhalb der IFD-Struktur                 *}
  {*              pos: Integer (Default 1)                              *}
  {*               = Auswahl des Auszugebenen Wertes                    *}
  {* Rckgabe:    Result:LongInt                                        *}
  {*               = Ausgabe vom Wert                                   *}
  {**********************************************************************}

  function GetTiffVal_I(i: Integer;pos: Integer=1):LongInt;
  Var Ext : Boolean;
      xPos: int64;
      xLen: Int64;
      TBuf: TBufType;
  begin
    Ext:=True;
    Result:=-1;
    if FBuf.TifTag[i].ICnt=1 then begin
      case FBuf.TIFTag[i].Typ of
        1 : begin
              Result:=Word(FBuf.TIFTag[i].BOf1);
              Ext:=False;
            end;
        3 : begin
              Result:=Word(FBuf.TIFTag[i].WOf1);
              Ext:=False;
            end;
        4 : begin
              Result:=FBuf.TIFTag[i].IOfs;
              Ext:=False;
            end;
      end;
    end;
    if Ext and (FBuf.TIFTag[i].Typ=3) then begin
      xPos:=FileSeek(FH,FBuf.TIFTag[i].IOfs,0);
      xLen:=FileRead(FH,TBuf,min(FBuf.TifTag[i].ICnt*2,100));
      if Mot then
        Result:=Swap(TBuf.WW[Pos-1])
      else
        Result:=TBuf.WW[Pos-1];
    end;
    if Ext and (FBuf.TIFTag[i].Typ=4) then begin
      xPos:=FileSeek(FH,FBuf.TIFTag[i].IOfs,0);
      xLen:=FileRead(FH,TBuf,min(FBuf.TifTag[i].ICnt*4,100));
      if Mot then
        Result:=LongW(TBuf.MW[Pos-1])
      else
        Result:=TBuf.IW[Pos-1];
    end;
  end;

begin
{
  StringGrid1.ColWidths[0]:=120;
  StringGrid1.ColWidths[1]:=120;
  StringGrid1.ColWidths[2]:=120;
}
  StringGrid1.cells[0,0]:='Sektion';
  StringGrid1.Cells[1,0]:='Feld';
  StringGrid1.Cells[2,0]:='Wert';
  Line:=1;
  StringGrid1.RowCount:=2;
  Mot:=False;
  if ((FBuf.DummyA[0]=$4D) or (FBuf.DummyA[0]=$49)) and
     (FBuf.DummyA[0]=FBuf.DummyA[1]) then begin
    cntP:=0;
    cntD:=0;
    if (FBuf.DummyA[0]=$4D) then begin
      Mot:=True;
      StringGrid1.cells[0,1]:='Hdr M';
      Adr:=FBuf.DummyA[7]+Word(FBuf.DummyA[6])*256;
      Adr:=Adr+FBuf.DummyA[5]*65536+FBuf.DummyA[4]*16777216;
    end
    else begin
      StringGrid1.cells[0,1]:='Hdr I';
      Adr:=FBuf.DummyA[4]+Word(FBuf.DummyA[5])*256;
      Adr:=Adr+FBuf.DummyA[6]*65536+FBuf.DummyA[7]*16777216;
    end;
    StringGrid1.Cells[1,1]:='Adr IFD1';
    StringGrid1.Cells[2,1]:=IntToHex(Adr,8);
    repeat
      Adr:=FileSeek(FH,Adr,0);
      Len:=FileRead(FH,FBuf.DummyP,512);
      inc(cntP);
      if Mot then SwapTIFF_IFD;
      StringGrid1.RowCount:=StringGrid1.RowCount+FBuf.TifCnt+2;
      inc(Line);
      StringGrid1.cells[0,Line]:='IFD';
      StringGrid1.Cells[1,Line]:='Anz.Tags';
      StringGrid1.Cells[2,Line]:=IntToStr(FBuf.TifCnt);
      for i:=1 to FBuf.TifCnt do begin
        inc(Line);
        StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
        StringGrid1.Cells[1,Line]:='Typ/Anz';
        StringGrid1.Cells[2,Line]:=IntToStr(FBuf.TIFTag[i].Typ)+'/'+
                                   IntToStr(FBuf.TIFTag[i].ICnt);
        case FBuf.TIFTag[i].Tag of
          $00FE: begin
                   case GetTiffVal_I(i) of
                      0 : HStr:='0 - normal';
                      1 : HStr:='1 - Thumbnail';
                      2 : HStr:='2 - Page';
                      3 : HStr:='3 - Page/Thumbnail';
                      4 : HStr:='4 - Tranparency mask';
                      5 : HStr:='5 - Transp/Thumbnail';
                      6 : HStr:='6 - Transp/Page';
                      7 : HStr:='7 - Transp/Page/Thumb';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='SubFile new';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $00FF: begin
                   case GetTiffVal_I(i) of
                      1 : HStr:='1 - full Resolution';
                      2 : HStr:='2 - Thumbnail';
                      3 : HStr:='3 - Single Page';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='SubFile old';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0100: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Imag. Width';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0101: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Imag. Heigh';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0102: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Bits p. Sample';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0103: begin
                   case GetTiffVal_I(i) of
                      1 : HStr:='1 - Uncompressed';
                      2 : HStr:='2 - CCITT 1D';
                      3 : HStr:='3 - Group 3 Fax';
                      4 : HStr:='4 - Group 4 Fax';
                      5 : HStr:='5 - LZW';
                      6 : HStr:='6 - JPEG';
                      32771 : HStr:='32771 - Uncompr. Word-Align';
                      32773 : HStr:='32773 - PackBits';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Compression';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0106: begin
                   case GetTiffVal_I(i) of
                      0 : HStr:='0 - White=0';
                      1 : HStr:='1 - Black=0';
                      2 : HStr:='2 - RGB';
                      3 : HStr:='3 - RGB Palette';
                      4 : HStr:='4 - Tranparency mask';
                      5 : HStr:='5 - CMYK';
                      6 : HStr:='6 - YCbCr';
                      8 : HStr:='8 - CIELab';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Photom.Int.';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0107: begin
                   case GetTiffVal_I(i) of
                      1 : HStr:='1 - Bilevel';
                      2 : HStr:='2 - Dithered';
                      3 : HStr:='3 - Error-Diffusion';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Thresholding';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0108: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='CellWidth';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0109: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='CellHight';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $010A: begin
                   case GetTiffVal_I(i) of
                      1 : HStr:='1 - High->Lo';
                      2 : HStr:='2 - Lo->High';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Byte FillOrder';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $010D: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Doc.Name';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $010E: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Image Desc.';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $010F: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Make';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0110: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Model';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0111: begin
                   cntD:=cntD+FBuf.TIFTag[i].ICnt;
                 end;
          $0112: begin
                   case GetTiffVal_I(i) of
                      0 : HStr:='1 - Top-Left';
                      1 : HStr:='2 - Top-Right';
                      2 : HStr:='3 - Bottom-Left';
                      3 : HStr:='4 - Bottom-Right';
                      4 : HStr:='5 - Left-Top';
                      5 : HStr:='6 - Right-Top';
                      6 : HStr:='7 - Right-Bottom';
                      8 : HStr:='8 - Left-Bottom';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Orientation';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0115: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Sampels per Pixel';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0116: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Rows per Strip';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0118: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Min.Sample-Value';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0119: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Max.Sample-Value';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $011A: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='XResolution';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $011B: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='YResolution';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $011C: begin
                   case GetTiffVal_I(i) of
                      1 : HStr:='1 - Pixel';
                      2 : HStr:='2 - Farbebene';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Planar Conf';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $011D: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='PageName';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $011E: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='XPosition';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $011F: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='YPosition';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0124: begin
                   case GetTiffVal_I(i) of
                      0 : HStr:='0 - Std 1-dim Kod';
                      1 : HStr:='1 - 2-dim Kod';
                      2 : HStr:='2 - Unkomprimiert 1-dim';
                      3 : HStr:='3 - Unkomprimiert 2-dim';
                      4 : HStr:='4 - Std 1-dim Kod - Byte Padded';
                      5 : HStr:='5 - 2-dim Kod - Byte Padded';
                      6 : HStr:='6 - Unkompr. 1-dim - Byte Padded';
                      7 : HStr:='7 - Unkompr. 2-dim - Byte Padded';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='T4/Group3-Opt';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0125: begin
                   case GetTiffVal_I(i) of
                      0 : HStr:='0 - Standard';
                      2 : HStr:='2 - Unkomprimiert';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='T6/Group4-Opt';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0128: begin
                   case GetTiffVal_I(i) of
                      1 : HStr:='1 - Relativ';
                      2 : HStr:='2 - inch';
                      3 : HStr:='3 - cm';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Res.Unit';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0129: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Seite/Gesamt';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i,2,'/');
                 end;
          $0131: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Software';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0132: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Date/Time';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $013B: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Artist';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $013D: begin
                   case GetTiffVal_I(i) of
                      1 : HStr:='1 - ohne Schema';
                      2 : HStr:='2 - horiz. Differenzierung';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Prediction';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $0140: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Palette';
                   StringGrid1.Cells[2,Line]:=IntToStr(Round(FBuf.TIFTag[i].ICnt/3))+
                                              ' Eintrge';
                 end;
          $0142: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Tile Width';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0143: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Tile Hight';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $0144: begin
                   cntD:=cntD+FBuf.TIFTag[i].ICnt;
                 end;
          $0153: begin
                   case GetTiffVal_I(i) of
                      1 : HStr:='1 - unsigned Word';
                      2 : HStr:='2 - signed Int (2-compl)';
                      3 : HStr:='3 - IEEE Fliekomma';
                      4 : HStr:='4 - nicht definiert';
                     else HStr:=IntToStr(GetTiffVal_I(i))+' - Unknown';
                   end;
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Sample Format';
                   StringGrid1.Cells[2,Line]:=HStr;
                 end;
          $8216: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Texture Format*';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $8217: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Wrap Mode*';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          $8298: begin
                   StringGrid1.RowCount:=StringGrid1.RowCount+1;
                   inc(Line);
                   StringGrid1.cells[0,Line]:=IntToHex(FBuf.TIFTag[i].Tag,4);
                   StringGrid1.Cells[1,Line]:='Copyright';
                   StringGrid1.Cells[2,Line]:=GetTiffStr(i);
                 end;
          else   begin
                   null;
                 end;
        end;
        null;
      end;
      inc(Line);
      StringGrid1.cells[0,Line]:='IFD';
      StringGrid1.Cells[1,Line]:='Adr IFDx';
      StringGrid1.Cells[2,Line]:=IntToHex(FBuf.TIFTAG[FBuf.TifCnt+1].IFH,8);
      Adr:=FBuf.TIFTAG[FBuf.TifCnt+1].IFH;
    until (FBuf.TIFTAG[FBuf.TifCnt+1].IFH=0) or (cntP>=16) or (Adr>=FLen);
    Label12.Caption:=intToStr(cntP)+' Directories ('+intToStr(cntD)+' Daten-Blcke)';
  end
  else StringGrid1.RowCount:=1;
end;

procedure TForm1.GetEXIFInfo(FH: Integer);
begin
  null;
end;

{************************************************************************}
{* Evt-Methode: Open1Click                                  (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      ffen des OpenDialogs (mit Filter) mit anschienden Auf- *}
{*              ruf der Methode GetFileInfo beim Bettigen des Men-    *}
{*              punktes (MainMenu/Datei/)Open                           *}
{* VCL-Event:   TForm1.Open1.OnClick                                    *}
{************************************************************************}

procedure TForm1.Open1Click(Sender: TObject);
begin
  OpenDialog1.Filter:='TIFF-Bild (*.tif,*.tiff)|*.tif;*.tiff';
  if OpenDialog1.Execute then begin
    GetFileInfo(OpenDialog1.FileName);
  end
end;

{************************************************************************}
{* Pr.-Methode: WMDROPFILES                                 (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Auswertung der Message WM_DROPFILES mit anschienden Auf-*}
{*              ruf der Methode GetFileInfo falls Extension stimmt      *}
{* Win-Message: WM_DROPFILES;                                           *}
{************************************************************************}

procedure TForm1.WMDROPFILES(var Msg: TMessage);
var
  i, count, size: integer;
  dropfile: PChar;
  FileName: AnsiString;
begin
  inherited;
  count := DragQueryFile(Msg.WParam, $FFFFFFFF, dropfile, 255);
  for i := 0 to count - 1 do
  begin
    size := DragQueryFile(Msg.WParam, i, nil, 0) + 1;
    dropfile := StrAlloc(size);
    try
      DragQueryFile(Msg.WParam, i, dropfile, size);
      FileName:=StrPas(dropfile);
      if (AnsiUpperCase(ExtractFileExt(FileName))='.TIF') or
         (AnsiUpperCase(ExtractFileExt(FileName))='.TIFF') then
        GetFileInfo(FileName);
    finally
      StrDispose(dropfile);
    end;
  end;
  DragFinish(Msg.WParam);
end;

{************************************************************************}
{* Evt-Methode: Close1Click                                 (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Schlieen des Forms beim Bettigen des Menpunktes Close*}
{* VCL-Event:   TForm1.Close1.OnClick                                   *}
{************************************************************************}

procedure TForm1.Close1Click(Sender: TObject);
begin
  Close;
end;

{************************************************************************}
{* Evt-Methode: FormCreate                                  (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Zustzliches Initialisieren vom File-Drag and Drop beim *}
{*              Erstellen des Forms                                     *}
{* VCL-Event:   TForm1.OnCreate                                         *}
{************************************************************************}

procedure TForm1.FormCreate(Sender: TObject);
begin
  DragAcceptFiles(Handle, true);
end;

end.
