{************************************************************************}
{************************************************************************}
{* Modul:       PNGChk_Frm.pas                                          *}
{************************************************************************}
{* Inhalt:      Hauptfenster mit Auswahl und Analyse von PNG-Dateien    *}
{************************************************************************}
{* Funktion:    Extraktion der Horizontal-, Vertikal- und Farb-Auflsung*}
{*              sowie Paletten- und Pack-Informationen fr Bilder,      *}
{*              Titel, Instrumentenzahl und Lnge fr Musik, Titel und  *}
{*              Betriebssystem fr Programme.                           *}
{************************************************************************}
{* Version:     0.1.0 B001                                              *}
{* Autor:       Thomas Mainka                                           *}
{* Datum:       25.Jun.2016                                             *}
{* Vernderung: Drag&Drop-Interface vom Explorer eingebaut              *}
{*              Kommentar-Header hinzugefgt                            *}
{*              Zhlung der IDAT-Tags hinzugefgt                       *}
{*              Datenausgabe fr div. Tags ergnzt                      *}
{************************************************************************}
{* Revision:    B000 Erste Version im SVN-Repository zum Aufbau des SK  *}
{************************************************************************}
{* Class:       TForm1                                                  *}
{* Methoden:    TForm1.GetFileInfo                                      *}
{*              TForm1.GetPNGInfo                                       *}
{*              TForm1.GetEXIFInfo(z.Z. leer)                           *}
{*              TForm1.WMDROPFILES                                      *}
{* HRoutinen:   LongI                                                   *}
{*              LongW                                                   *}
{************************************************************************}

unit PNGChk_Frm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Menus, StdCtrls, Grids, Math, ShellApi;

type
  Byte4     = Array[1..4] of Byte;
  PngT      = packed record
                case Integer of
                  0 : (B: Byte4;);
                  1 : (C: Array[0..3] of AnsiChar;);
                  2 : (L: LongWord;);
              end;
  FBufType  = packed record
                case Integer of
                  -1: (DummyP: Array[0..4095] of AnsiChar;);
                   0: (DummyA: Array[0..4095] of Byte;);
                   1: (JFIFKen: Word;
                       JFIFLen: Word;
                       DummyB: Array[0..4091] of AnsiChar;);
                   2: (FFE0Ken: Word;
                       FFE0Len: Word;
                       FFE0Ide: Array[0..4] of AnsiChar;
                       FFE0Ver: Word;
                       FFE0AEi: Byte;
                       FFE0HAu: Word;
                       FFE0VAu: Word;
                       FFE0VHo: Byte;
                       FFE0VVe: Byte;
                       FFE0Dum: Array[14..4091] of AnsiChar;);
                   3: (FFC0Ken: Word;
                       FFC0Len: Word;
                       FFC0Pre: Byte;
                       FFC0Hor: Word;
                       FFC0Ver: Word;
                       FFC0CCp: Byte;
                       FFC0Id1: Byte;
                       FFC0SF1: Byte;
                       FFC0QT1: Byte;
                       FFC0Id2: Byte;
                       FFC0SF2: Byte;
                       FFC0QT2: Byte;
                       FFC0Id3: Byte;
                       FFC0SF3: Byte;
                       FFC0QT3: Byte;
                       FFC0Dum: Array[15..4091] of AnsiChar;);
                   4: (FFC4Ken: Word;
                       FFC4Len: Word;
                       FFC4CCp: Byte;
                       FFC4HTL: Array[0..15] of Byte;
                       FFC4HTa: Array[17..4091] of Byte;);
                   5: (FFCCKen: Word;
                       FFCCLen: Word;
                       FFCCCCp: Byte;
                       FFCCWAC: Byte;
                       FFCCDum: Array[2..4091] of AnsiChar;);
                   6: (FFDDKen: Word;
                       FFDDLen: Word;
                       FFDDInt: Word;
                       FFDDDum: Array[2..4091] of AnsiChar;);
                  40: (PNGFLen: Byte4;
                       PNGFKen: PngT;
                       PNGFDum: Array[0..4000] of AnsiChar;);
                  41: (IHDRLen: Byte4;
                       IHDRKen: PngT;
                       IHDRSiX: Byte4;
                       IHDRSiY: Byte4;
                       IHDRBpP: Byte;
                       IHDRPTy: Byte;
                       IHDRCom: Byte;
                       IHFRFil: Byte;
                       IHDRInt: Byte;
                       IHDRDum: Array[17..4091] of AnsiChar;);
                  42: (pHYsLen: Byte4;
                       pHYsKen: PngT;
                       pHYsSiX: Byte4;
                       pHYsSiY: Byte4;
                       pHYsUni: Byte;
                       pHYsDum: Array[13..4091] of AnsiChar;);
                  43: (cHRMLen: Byte4;
                       cHRMKen: PngT;
                       cHRMWPX: Byte4;
                       cHRMWPY: Byte4;
                       cHRMReX: Byte4;
                       cHRMReY: Byte4;
                       cHRMGrX: Byte4;
                       cHRMGrY: Byte4;
                       cHRMBlX: Byte4;
                       cHRMBlY: Byte4;
                       cHRMDum: Array[36..4091] of AnsiChar;);
                  44: (tIMELen: Byte4;
                       tIMEKen: PngT;
                       tIMEYea: Word;
                       tIMEMon: Byte;
                       tIMEDay: Byte;
                       tIMEHou: Byte;
                       tIMEMin: Byte;
                       tIMRSek: Byte;
                       tIMEDum: Array[17..4091] of AnsiChar;);
             end;
  TForm1 = class(TForm)
    MainMenu1: TMainMenu;
    OpenDialog1: TOpenDialog;
    StringGrid1: TStringGrid;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    Label8: TLabel;
    Label9: TLabel;
    Label10: TLabel;
    Datei1: TMenuItem;
    Open1: TMenuItem;
    Close1: TMenuItem;
    Label11: TLabel;
    Label12: TLabel;
    procedure Open1Click(Sender: TObject);
    procedure Close1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    { Private-Deklarationen }
    Line: Integer;
    FBuf: FBufType;
    FLen: int64;
    procedure GetFileInfo(FileName:string);
    procedure GetPNGInfo(FH:integer);
    procedure GetEXIFInfo(FH:integer);
    procedure WMDROPFILES(var Msg: TMessage); message WM_DROPFILES;  public
    { Public-Deklarationen }
  end;

var
  Form1: TForm1;

implementation

{$R *.dfm}

{************************************************************************}
{* Routine:     LongI                                       (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Umwandlung eines Byte4-Langwortes in ein Lang-Int       *}
{* Parameter:   x:Byte4                                                 *}
{*               = 4-Byte Array in Big-Endian Folge                     *}
{* Rckgabe:    Result:LongWord                                         *}
{*               = Longint (little-Endian)                              *}
{************************************************************************}

Function LongI(x:Byte4):Longint;
begin
   LongI:=x[4]+Word(x[3])*256+x[2]*65536+x[1]*16777216;
end;

{************************************************************************}
{* Routine:     LongW                                       (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Umwandlung eines Byte4-Langwortes in ein Lang-Wort      *}
{* Parameter:   x:Byte4                                                 *}
{*               = 4-Byte Array in Big-Endian Folge                     *}
{* Rckgabe:    Result:LongWord                                         *}
{*               = LongWord (little-Endian)                             *}
{************************************************************************}

Function LongW(x:Byte4):LongWord;
begin
   LongW:=x[4]+Word(x[3])*256+x[2]*65536+x[1]*16777216;
end;

{************************************************************************}
{* Pr.-Methode: GetFileInfo                                 (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Ermitteln und Anzeige der Datei-Eigenschaften, ffnen   *}
{*              der Datei und Aufruf der Typ-Spezifischen Analyse-      *}
{*              Routine (hier GetPNGInfo)                               *}
{* Parameter:   FileName:string                                         *}
{*               = kompletter Dateiname (incl Pfad)                     *}
{************************************************************************}

procedure TForm1.GetFileInfo(FileName:string);
Var FH   : integer;
    Addr : int64;
    Attr : integer;
    AStr : Array[0..7] of AnsiChar;
    HStr : Array[0..255] of AnsiChar;
begin
  Label2.Caption:=ExtractFilePath(FileName);
  Label4.Caption:=ExtractFileName(FileName);
  Attr:=FileGetAttr(FileName);
  AStr:='--------';
  if (Attr and faReadOnly) > 0 then AStr[7]:='R';
  if (Attr and faHidden) > 0   then AStr[6]:='H';
  if (Attr and faSysFile) > 0  then AStr[5]:='S';
  if (Attr and faArchive) > 0  then AStr[3]:='A';
  if (Attr and faSymLink) > 0  then AStr[2]:='J';
  Label8.Caption:=AStr;
  Label10.Caption:=DateTimeToStr(FileDateToDateTime(FileAge(FileName)));
  StrPCopy(HStr,FileName);
  FH:=FileOpen(HStr,of_Read+of_Share_Compat);
  if (FH=-1) then begin
    FH:=FileOpen(HStr,of_Read+of_Share_Deny_none);
  end;
  if FH>0 then begin
    FLen:=FileSeek(FH,0,2);
    Label6.Caption:=InttoStr(FLen)+' Bytes';
    Addr:=FileSeek(FH,0,0);
    Addr:=Addr+FileRead(FH,FBuf.DummyP,8);
    GetPNGInfo(FH);
    FileClose(FH);
  end
  else begin
    Label12.Caption:='File could not opened';
  end;
end;

{************************************************************************}
{* Pr.-Methode: GetPNGInfo                                  (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Ermitteln und Anzeige der Datei-Eigenschaften, ffnen   *}
{*              der Datei und Aufruf der Typ-Spezifischen Analyse-      *}
{*              Routine (hier GetTIFFInfo)                              *}
{* Parameter:   FileName:string                                         *}
{*               = kompletter Dateiname (incl Pfad)                     *}
{************************************************************************}

procedure TForm1.GetPNGInfo(FH: Integer);
Var Adr : Int64;
    Len : integer;
    CLen: Int64;
    HStr: AnsiString;
    PTyp: integer;
    sPos: integer;
    sChr: AnsiChar;
    cntP: integer;
    cntD: integer;

begin
{
  StringGrid1.ColWidths[0]:=120;
  StringGrid1.ColWidths[1]:=120;
  StringGrid1.ColWidths[2]:=120;
}
  StringGrid1.cells[0,0]:='Sektion';
  StringGrid1.Cells[1,0]:='Feld';
  StringGrid1.Cells[2,0]:='Wert';
  Line:=0;
  if (FBuf.DummyA[0]=$89) and (FBuf.DummyA[1]=$50) and
     (FBuf.DummyA[2]=$4E) and (FBuf.DummyA[3]=$47) and
     (FBuf.DummyA[4]=$0D) and (FBuf.DummyA[5]=$0A) and
     (FBuf.DummyA[6]=$1A) and (FBuf.DummyA[7]=$0A) then begin
    cntP:=0;
    cntD:=0;
    Adr:=8;
    repeat
      Len:=FileRead(FH,FBuf.DummyP,8);
      CLen:=LongI(FBuf.IHDRLen);
      case FBuf.IHDRKen.L of
        $52444849 : begin
                      cntP:=1;
                      Len:=FileRead(FH,FBuf.PNGFDum,min(CLen,4000));
                      StringGrid1.RowCount:=(Line+6)+1;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='IHDR';
                      StringGrid1.Cells[1,Line]:='Typ';
                      StringGrid1.Cells[2,Line]:='Image-Header';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='IHDR';
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='IHDR';
                      StringGrid1.Cells[1,Line]:='Auflsung';
                      StringGrid1.Cells[2,Line]:=IntToStr(LongI(FBuf.IHDRSiX))+'x'+
                                                 IntToStr(LongI(FBuf.IHDRSiY));
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='IHDR';
                      StringGrid1.Cells[1,Line]:='Farbtiefe/Kanal';
                      StringGrid1.Cells[2,Line]:=IntToStr(FBuf.IHDRBpP);
                      PTyp:=FBuf.IHDRPTy;  {Speichern fr sptere Tags}
                      case FBuf.IHDRPTy of
                         0 : HStr:='0 - Graustufe';
                         2 : HStr:='2 - Farbe';
                         3 : HStr:='3 - Farbe/Palette';
                         4 : HStr:='4 - Graustufe+Alpha';
                         6 : HStr:='6 - Farbe+Alpha';
                        else HStr:=IntToStr(FBuf.IHDRPTy)+' - Unbekannt';
                      end;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='IHDR';
                      StringGrid1.Cells[1,Line]:='Farb-Typ';
                      StringGrid1.Cells[2,Line]:=HStr;
                      case FBuf.IHDRInt of
                         0 : HStr:='0 - ohne';
                         1 : HStr:='1 - Adam7';
                        else HStr:=IntToStr(FBuf.IHDRInt)+' - Unbekannt';
                      end;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='IHDR';
                      StringGrid1.Cells[1,Line]:='Interlace-Verf.';
                      StringGrid1.Cells[2,Line]:=HStr;
                    end;
        $4D524863 : begin {cHRM}
                      Len:=FileRead(FH,FBuf.PNGFDum,min(CLen,4000));
                      StringGrid1.RowCount:=(Line+6)+1;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='cHRM';
                      StringGrid1.Cells[1,Line]:='Typ';
                      StringGrid1.Cells[2,Line]:='Prim.Chromacities';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='cHRM';
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='cHRM';
                      StringGrid1.Cells[1,Line]:='CIE White-Point';
                      StringGrid1.Cells[2,Line]:='['+
                                     FloatToStr(LongI(FBuf.cHRMWPX)/100000)+' / '+
                                     FloatToStr(LongI(FBuf.cHRMWPY)/100000)+']';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='cHRM';
                      StringGrid1.Cells[1,Line]:='CIE 1931 Red';
                      StringGrid1.Cells[2,Line]:='['+
                                     FloatToStr(LongI(FBuf.cHRMReX)/100000)+' / '+
                                     FloatToStr(LongI(FBuf.cHRMReY)/100000)+']';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='cHRM';
                      StringGrid1.Cells[1,Line]:='CIE 1931 Blue';
                      StringGrid1.Cells[2,Line]:='['+
                                     FloatToStr(LongI(FBuf.cHRMBlX)/100000)+' / '+
                                     FloatToStr(LongI(FBuf.cHRMBlY)/100000)+']';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='cHRM';
                      StringGrid1.Cells[1,Line]:='CIE 1931 Green';
                      StringGrid1.Cells[2,Line]:='['+
                                     FloatToStr(LongI(FBuf.cHRMGrX)/100000)+' / '+
                                     FloatToStr(LongI(FBuf.cHRMGrY)/100000)+']';
                    end;
        $42475273 : begin
                      cntP:=1;
                      Len:=FileRead(FH,FBuf.PNGFDum,min(CLen,4000));
                      StringGrid1.RowCount:=(Line+3)+1;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='sRGB';
                      StringGrid1.Cells[1,Line]:='Typ';
                      StringGrid1.Cells[2,Line]:='Stand. RGB Farbraum';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='sRGB';
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                      case FBuf.DummyA[8] of
                         0 : HStr:='0 - Wahrnehmung';
                         1 : HStr:='1 - Relativ Colorimetrisch';
                         2 : HStr:='2 - Sttigung';
                         3 : HStr:='3 - Absolut Colorimetrisch';
                        else HStr:=IntToStr(FBuf.DummyA[8])+' - Unbekannt';
                      end;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='sRGB';
                      StringGrid1.Cells[1,Line]:='ICC intent';
                      StringGrid1.Cells[2,Line]:=HStr;
                    end;
        $414D4167 : begin {gAMA}
                      Len:=FileRead(FH,FBuf.PNGFDum,min(CLen,4000));
                      StringGrid1.RowCount:=(Line+3)+1;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='gAMA';
                      StringGrid1.Cells[1,Line]:='Typ';
                      StringGrid1.Cells[2,Line]:='Image Gamma';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='gAMA';
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='gAMA';
                      StringGrid1.Cells[1,Line]:='Koeffizient';
                      StringGrid1.Cells[2,Line]:=FloatToStr(LongI(FBuf.cHRMWPX)/100000);
                    end;
        $73594870 : begin {pHYs}
                      Len:=FileRead(FH,FBuf.PNGFDum,min(CLen,4000));
                      inc(Line);
                      StringGrid1.RowCount:=(Line+6)+1;
                      StringGrid1.Cells[0,Line]:='pHYs';
                      StringGrid1.Cells[1,Line]:='Typ';
                      StringGrid1.Cells[2,Line]:='Phys.Pixelgr.';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='pHYs';
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='pHYs';
                      StringGrid1.Cells[1,Line]:='X-Auflsung';
                      StringGrid1.Cells[2,Line]:=IntToStr(LongW(FBuf.pHYsSiX));
                      if (FBuf.pHYsUni=1) then
                        StringGrid1.Cells[2,Line]:=
                            StringGrid1.Cells[2,Line]+
                            ' dpm (' +
                            IntToStr(Round(LongW(FBuf.pHYsSiX)/1000*25.4))+
                            ' dpi)';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='pHYs';
                      StringGrid1.Cells[1,Line]:='Y-Auflsung';
                      StringGrid1.Cells[2,Line]:=IntToStr(LongW(FBuf.pHYsSiY));
                      if (FBuf.pHYsUni=1) then
                        StringGrid1.Cells[2,Line]:=
                               StringGrid1.Cells[2,Line]+
                               ' dpm (' +
                               IntToStr(Round(LongW(FBuf.pHYsSiX)/1000*25.4))+
                               ' dpi)';
                    end;
        $54494273 : begin {sBIT}
                      Len:=FileRead(FH,FBuf.PNGFDum,min(CLen,4000));
                      inc(Line);
                      StringGrid1.RowCount:=(Line+3)+1;
                      StringGrid1.Cells[0,Line]:='sBIT';
                      StringGrid1.Cells[1,Line]:='Typ';
                      StringGrid1.Cells[2,Line]:='sign. Bits';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='sBIT';
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='sBIT';
                      case PTyp of
                        0 : begin
                              StringGrid1.Cells[1,Line]:='GS';
                              StringGrid1.Cells[2,Line]:=intToStr(FBuf.DummyA[8]);
                            end;
                        2,3 : begin
                              StringGrid1.Cells[1,Line]:='R/G/B';
                              StringGrid1.Cells[2,Line]:=
                                intToStr(FBuf.DummyA[8])+'/'+
                                intToStr(FBuf.DummyA[9])+'/'+
                                intToStr(FBuf.DummyA[10]);
                            end;
                        4 : begin
                              StringGrid1.Cells[1,Line]:='GS-A';
                              StringGrid1.Cells[2,Line]:=
                                intToStr(FBuf.DummyA[8])+'-'+
                                intToStr(FBuf.DummyA[9]);
                            end;
                        6 : begin
                              StringGrid1.Cells[1,Line]:='R/G/B-A';
                              StringGrid1.Cells[2,Line]:=
                                intToStr(FBuf.DummyA[8])+'/'+
                                intToStr(FBuf.DummyA[9])+'/'+
                                intToStr(FBuf.DummyA[10])+'-'+
                                intToStr(FBuf.DummyA[11]);
                            end;
                      end;
                    end;
        $74584574 : begin {tEXt}
                      Len:=FileRead(FH,FBuf.PNGFDum,min(CLen,4000));
                      inc(Line);
                      StringGrid1.RowCount:=(Line+3)+1;
                      StringGrid1.Cells[0,Line]:='tEXt';
                      StringGrid1.Cells[1,Line]:='Typ';
                      StringGrid1.Cells[2,Line]:='Unkompr. Text';
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='tEXt';
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                      inc(Line);
                      StringGrid1.Cells[0,Line]:='tEXt';
                      StringGrid1.Cells[1,Line]:='';
                      sPos:=0;
                      while (sPos<CLen) and (FBuf.PNGFDum[sPos]<>chr($00)) do begin
                        StringGrid1.Cells[1,Line]:=StringGrid1.Cells[1,Line]+
                                                   FBuf.PNGFDum[sPos];
                        inc(sPos);
                      end;
                      StringGrid1.Cells[2,Line]:='';
                      inc(sPos);
                      while (sPos<CLen) and (FBuf.PNGFDum[sPos]<>chr($00)) do begin
                        StringGrid1.Cells[2,Line]:=StringGrid1.Cells[2,Line]+
                                                   FBuf.PNGFDum[sPos];
                        inc(sPos);
                      end;
                    end;
        $54414449 : begin  {IDAT}
                      inc(cntD);
                      StringGrid1.RowCount:=(Line+1)+1;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:=FBuf.IHDRKen.C;
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                    end;
        else        begin
                      StringGrid1.RowCount:=(Line+1)+1;
                      inc(Line);
                      StringGrid1.Cells[0,Line]:=FBuf.IHDRKen.C;
                      StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                      StringGrid1.Cells[2,Line]:=IntToStr(CLen)+'/'+
                                                 IntToHex(Adr+12+CLen,8);
                    end;
      end;
      Adr:=Adr+12+CLen;
      Adr:=FileSeek(FH,Adr,0);
      Label12.Caption:=IntToHex(Adr,8);
    until (Adr>=FLen) or (FBuf.IHDRKen.L=$444E4549);
    if (FBuf.IHDRKen.L=$444E4549) then
      Label12.Caption:=intToStr(cntP)+' Frames ('+intToStr(cntD)+' IDAT-Blcke)';
  end;
end;

procedure TForm1.GetEXIFInfo(FH: Integer);
begin
  null;
end;

{************************************************************************}
{* Evt-Methode: Open1Click                                  (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      ffen des OpenDialogs (mit Filter) mit anschienden Auf- *}
{*              ruf der Methode GetFileInfo beim Bettigen des Men-    *}
{*              punktes (MainMenu/Datei/)Open                           *}
{* VCL-Event:   TForm1.Open1.OnClick                                    *}
{************************************************************************}

procedure TForm1.Open1Click(Sender: TObject);
begin
  OpenDialog1.Filter:='PNG-Bild (*.png)|*.png';
  if OpenDialog1.Execute then begin
    GetFileInfo(OpenDialog1.FileName);
  end
end;

{************************************************************************}
{* Pr.-Methode: WMDROPFILES                                 (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Auswertung der Message WM_DROPFILES mit anschienden Auf-*}
{*              ruf der Methode GetFileInfo falls Extension stimmt      *}
{* Win-Message: WM_DROPFILES;                                           *}
{************************************************************************}

procedure TForm1.WMDROPFILES(var Msg: TMessage);
var
  i, count, size: integer;
  dropfile: PChar;
  FileName: AnsiString;
begin
  inherited;
  count := DragQueryFile(Msg.WParam, $FFFFFFFF, dropfile, 255);
  for i := 0 to count - 1 do
  begin
    size := DragQueryFile(Msg.WParam, i, nil, 0) + 1;
    dropfile := StrAlloc(size);
    try
      DragQueryFile(Msg.WParam, i, dropfile, size);
      FileName:=StrPas(dropfile);
      if AnsiUpperCase(ExtractFileExt(FileName))='.PNG' then
        GetFileInfo(FileName);
    finally
      StrDispose(dropfile);
    end;
  end;
  DragFinish(Msg.WParam);
end;

{************************************************************************}
{* Evt-Methode: Close1Click                                 (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Schlieen des Forms beim Bettigen des Menpunktes Close*}
{* VCL-Event:   TForm1.Close1.OnClick                                   *}
{************************************************************************}

procedure TForm1.Close1Click(Sender: TObject);
begin
  Close;
end;

{************************************************************************}
{* Evt-Methode: FormCreate                                  (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Zustzliches Initialisieren vom File-Drag and Drop beim *}
{*              Erstellen des Forms                                     *}
{* VCL-Event:   TForm1.OnCreate                                         *}
{************************************************************************}

procedure TForm1.FormCreate(Sender: TObject);
begin
  DragAcceptFiles(Handle, true);
end;

end.
