{************************************************************************}
{************************************************************************}
{* Modul:       JPGChk_Frm.pas                                          *}
{************************************************************************}
{* Inhalt:      Hauptfenster mit Auswahl und Analyse von PNG-Dateien    *}
{************************************************************************}
{* Funktion:    Extraktion der Horizontal-, Vertikal- und Farb-Auflsung*}
{*              sowie Paletten- und Pack-Informationen fr Bilder,      *}
{*              Titel, Instrumentenzahl und Lnge fr Musik, Titel und  *}
{*              Betriebssystem fr Programme.                           *}
{************************************************************************}
{* Version:     0.1.0 B001                                              *}
{* Autor:       Thomas Mainka                                           *}
{* Datum:       29.Jun.2016                                             *}
{* Vernderung: Drag&Drop-Interface vom Explorer eingebaut              *}
{*              Kommentar-Header hinzugefgt                            *}
{*              Ausgabe der Farb-Komponenten Inf. aus dem SOFx-Block    *}
{*              Ausgabe der Anzahl der Datenblcke in Label12           *]
{************************************************************************}
{* Revision:    B000 Erste Version im SVN-Repository zum Aufbau des SK  *}
{************************************************************************}
{* Class:       TForm1                                                  *}
{* Methoden:    TForm1.GetFileInfo                                      *}
{*              TForm1.Parse_Scan                                       *}
{*              TForm1.Fill_CxInfo                                      *}
{*              TForm1.GetJFIFInfo                                      *}
{*              TForm1.GetEXIFInfo(z.Z. leer)                           *}
{*              TForm1.WMDROPFILES                                      *}
{************************************************************************}

unit JPGChk_Frm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Menus, StdCtrls, Grids, Math, ShellApi;

type
  FBufType  = packed record
                case Integer of
                  -1: (DummyP: Array[0..4095] of AnsiChar;);
                   0: (DummyA: Array[0..4095] of Byte;);
                   1: (JFIFKen: Word;
                       JFIFLen: Word;
                       DummyB: Array[0..4091] of AnsiChar;);
                   2: (FFE0Ken: Word;
                       FFE0Len: Word;
                       FFE0Ide: Array[0..4] of AnsiChar;
                       FFE0Ver: Word;
                       FFE0AEi: Byte;
                       FFE0HAu: Word;
                       FFE0VAu: Word;
                       FFE0VHo: Byte;
                       FFE0VVe: Byte;
                       FFE0Dum: Array[14..4091] of AnsiChar;);
                   3: (FFC0Ken: Word;
                       FFC0Len: Word;
                       FFC0Pre: Byte;
                       FFC0Hor: Word;
                       FFC0Ver: Word;
                       FFC0CCp: Byte;
                       FFC0Id1: Byte;
                       FFC0SF1: Byte;
                       FFC0QT1: Byte;
                       FFC0Id2: Byte;
                       FFC0SF2: Byte;
                       FFC0QT2: Byte;
                       FFC0Id3: Byte;
                       FFC0SF3: Byte;
                       FFC0QT3: Byte;
                       FFC0Dum: Array[15..4091] of AnsiChar;);
                   4: (FFC4Ken: Word;
                       FFC4Len: Word;
                       FFC4CCp: Byte;
                       FFC4HTL: Array[0..15] of Byte;
                       FFC4HTa: Array[17..4091] of Byte;);
                   5: (FFCCKen: Word;
                       FFCCLen: Word;
                       FFCCCCp: Byte;
                       FFCCWAC: Byte;
                       FFCCDum: Array[2..4091] of AnsiChar;);
                   6: (FFDDKen: Word;
                       FFDDLen: Word;
                       FFDDInt: Word;
                       FFDDDum: Array[2..4091] of AnsiChar;);
             end;
  TForm1 = class(TForm)
    MainMenu1: TMainMenu;
    OpenDialog1: TOpenDialog;
    StringGrid1: TStringGrid;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    Label8: TLabel;
    Label9: TLabel;
    Label10: TLabel;
    Datei1: TMenuItem;
    Open1: TMenuItem;
    Close1: TMenuItem;
    Label11: TLabel;
    Label12: TLabel;
    procedure Open1Click(Sender: TObject);
    procedure Close1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    { Private-Deklarationen }
    Line: Integer;
    FBuf: FBufType;
    FLen: int64;
    procedure GetFileInfo(FileName:string);
    procedure GetJFIFInfo(FH:integer);
    procedure GetEXIFInfo(FH:integer);
    procedure Parse_Scan(FH: Integer; Var Adr: int64);
    procedure Fill_CxInfo(Var Line:Integer; Adr: Int64; Tag, Desc: AnsiString);
    procedure WMDROPFILES(var Msg: TMessage); message WM_DROPFILES;
  public
    { Public-Deklarationen }
  end;

var
  Form1: TForm1;

implementation

{$R *.dfm}

{************************************************************************}
{* Pr.-Methode: GetFileInfo                                 (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Ermitteln und Anzeige der Datei-Eigenschaften, ffnen   *}
{*              der Datei und Aufruf der Typ-Spezifischen Analyse-      *}
{*              Routine (hier GetJFIFInfo)                              *}
{* Parameter:   FileName:string                                         *}
{*               = kompletter Dateiname (incl Pfad)                     *}
{************************************************************************}

procedure TForm1.GetFileInfo(FileName:string);
Var FH   : integer;
    Addr : int64;
    Attr : integer;
    AStr : Array[0..7] of AnsiChar;
    HStr : Array[0..255] of AnsiChar;
begin
  Label2.Caption:=ExtractFilePath(FileName);
  Label4.Caption:=ExtractFileName(FileName);
  Attr:=FileGetAttr(FileName);
  AStr:='--------';
  if (Attr and faReadOnly) > 0 then AStr[7]:='R';
  if (Attr and faHidden) > 0   then AStr[6]:='H';
  if (Attr and faSysFile) > 0  then AStr[5]:='S';
  if (Attr and faArchive) > 0  then AStr[3]:='A';
  if (Attr and faSymLink) > 0  then AStr[2]:='J';
  Label8.Caption:=AStr;
  Label10.Caption:=DateTimeToStr(FileDateToDateTime(FileAge(FileName)));
  StrPCopy(HStr,FileName);
  FH:=FileOpen(HStr,of_Read+of_Share_Compat);
  if (FH=-1) then begin
    FH:=FileOpen(HStr,of_Read+of_Share_Deny_none);
  end;
  if FH>0 then begin
    FLen:=FileSeek(FH,0,2);
    Label6.Caption:=InttoStr(FLen)+' Bytes';
    Addr:=FileSeek(FH,0,0);
    Addr:=Addr+FileRead(FH,FBuf.DummyP,2);
    GetJFIFInfo(FH);
    FileClose(FH);
  end
  else begin
    Label12.Caption:='File could not opened';
  end;
end;

{************************************************************************}
{* Pr.-Methode: Parse_Scan                                  (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Lesen eines Bilddaten-Bereichs (Scan) und Ermittlung von*}
{*              dessen Lnge (Position des nchsten Tags)               *}
{* Parameter:   FH:Integer                                              *}
{*               = FileHandle                                           *}
{*              Adr:Int64 (IN/OUT)                                      *}
{*               = (Start/End-)Position innerhalb der Datei             *}
{************************************************************************}

Procedure TForm1.Parse_Scan(FH: Integer; Var Adr: int64);
Var Len: integer;
    Pos: integer;
    Buffer: Array[0..1023] of Byte;    
begin
  repeat
    Adr:=FileSeek(FH,Adr,0);
    Len:=FileRead(FH,Buffer,1024);
    if Len<>0 then begin
      Pos:=-1;
      repeat
        Inc(Pos);
      until ((Buffer[Pos]=$ff) and (Buffer[Pos+1]<>$00)) or (Pos=Min(Len,1020));
      Adr:=Adr+Pos;
    end;
  until ((Buffer[Pos]=$ff) and (Buffer[Pos+1]<>$00)) or (Pos=Len);
end;

{************************************************************************}
{* Pr.-Methode: Fill_CxInfo                                 (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Ausgabe der Informationen bezglich der gleichartig     *}
{*              aufgebauten StartOfFrame-Tags fr die unterschiedlichen *}
{*              Codierungs/Kompressions-Verfahren im StringGrid         *}
{* Parameter:   Line:integer (IN/OUT)                                   *}
{*               = aktuelle Zeile im StrinGrid                          *}
{*              Adr:Int64                                               *}
{*               = auszugebene (Start-)Position in der Datei            *}
{*              Tag:AnsiString                                          *}
{*               = auszugebene Tag-Bezeichnung                          *}
{*              Desc:AnsiString                                         *}
{*               = auszugebene Beschreibung (Typ/Kompr)                 *}
{************************************************************************}

Procedure TForm1.Fill_CxInfo(Var Line:Integer; Adr:Int64; Tag, Desc: AnsiString);
begin
  StringGrid1.RowCount:=(Line+6)+1;
  inc(Line);
  StringGrid1.Cells[0,Line]:=Tag;
  StringGrid1.Cells[1,Line]:='Typ';
  StringGrid1.Cells[2,Line]:=Desc;
  inc(Line);
  StringGrid1.Cells[0,Line]:=Tag;
  StringGrid1.Cells[1,Line]:='Lnge/EndPos';
  StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                             IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
  inc(Line);
  StringGrid1.Cells[0,Line]:=Tag;
  StringGrid1.Cells[1,Line]:='DataPrec';
  StringGrid1.Cells[2,Line]:=IntToStr(FBuf.FFC0Pre);
  inc(Line);
  StringGrid1.Cells[0,Line]:=Tag;
  StringGrid1.Cells[1,Line]:='Gre';
  StringGrid1.Cells[2,Line]:=IntToStr(swap(FBuf.FFC0Hor))+'x'+
                             IntToStr(swap(FBuf.FFC0Ver));
  inc(Line);
  StringGrid1.Cells[0,Line]:=Tag;
  StringGrid1.Cells[1,Line]:='Farb-Komp';
  StringGrid1.Cells[2,Line]:=IntToStr(FBuf.FFC0CCp);
  inc(Line);
  StringGrid1.Cells[0,Line]:=Tag;
  StringGrid1.Cells[1,Line]:='Id 1/Abt.HV/Nr.QT';
  StringGrid1.Cells[2,Line]:=IntToStr(FBuf.FFC0Id1)+'/'+IntToHex(FBuf.FFC0SF1,2)+'/'+
                             IntToStr(FBuf.FFC0QT1);
  if FBuf.FFC0CCp=3 then begin
    StringGrid1.RowCount:=(Line+2)+1;
    inc(Line);
    StringGrid1.Cells[0,Line]:=Tag;
    StringGrid1.Cells[1,Line]:='Id 2/Abt.HV/Nr.QT';
    StringGrid1.Cells[2,Line]:=IntToStr(FBuf.FFC0Id2)+'/'+IntToHex(FBuf.FFC0SF2,2)+'/'+
                               IntToStr(FBuf.FFC0QT2);
    inc(Line);
    StringGrid1.Cells[0,Line]:=Tag;
    StringGrid1.Cells[1,Line]:='Id 3/Abt.HV/Nr.QT';
    StringGrid1.Cells[2,Line]:=IntToStr(FBuf.FFC0Id3)+'/'+IntToHex(FBuf.FFC0SF3,2)+'/'+
                               IntToStr(FBuf.FFC0QT3);
  end;
end;

{************************************************************************}
{* Pr.-Methode: GetJFIFInfo                                 (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Analyse und Ausgabe der JFIF-JPG-Tags im StringGrid     *}
{* Parameter:   FH:Integer                                              *}
{*               = FileHandel                                           *}
{************************************************************************}

procedure TForm1.GetJFIFInfo(FH: Integer);
Var Adr : Int64;
    Len : integer;
    HStr: AnsiString;
    cntP: integer;
    cntD: integer;

  {**********************************************************************}
  {* Routine:     GetFBufStr                                (0.1.0 B000)*}
  {**********************************************************************}
  {* Inhalt:      Ermittlung eines 0-delimiten Strings bis zur max Lnge*}
  {*              bzw. Buffergre                                      *}
  {* Rckgabe:    Result:Ansistring                                     *}
  {*               = Ausgabe vom String                                 *}
  {**********************************************************************}

  Function GetFBufStr:AnsiString;
  Var i: integer;
  begin
    Result:='';
    i:=0;
    while (i<min(swap(FBuf.JFIFLen),4000)) and (FBuf.DummyA[i+4]<>$00) do begin
      Result:=Result+FBuf.DummyB[i];
      inc(i);
    end;
  end;

begin
{
  StringGrid1.ColWidths[0]:=120;
  StringGrid1.ColWidths[1]:=120;
  StringGrid1.ColWidths[2]:=120;
}
  StringGrid1.cells[0,0]:='Sektion';
  StringGrid1.Cells[1,0]:='Feld';
  StringGrid1.Cells[2,0]:='Wert';
  Line:=1;
  if FBuf.JFIFKen=$d8ff then begin
    cntP:=0;
    cntD:=0;
    StringGrid1.Cells[0,1]:='FFD8 SOI';
    StringGrid1.Cells[1,1]:='Typ';
    StringGrid1.Cells[2,1]:='Start of Image (Datei Anfang)';
    Adr:=2;
    repeat
      Len:=FileRead(FH,FBuf.DummyP,4);
      case FBuf.JFIFKen of
        $d9ff: begin
                 StringGrid1.RowCount:=(Line+1)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFD9 EOI';
                 StringGrid1.Cells[1,Line]:='Typ';
                 StringGrid1.Cells[2,Line]:='End of Image (Datei Ende)';
               end;
        $daff: begin
                 StringGrid1.RowCount:=(Line+2)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFDA SOS';
                 StringGrid1.Cells[1,Line]:='Typ';
                 StringGrid1.Cells[2,Line]:='Start of Scan (Bilddaten)';
                 Adr:=Adr+2;
                 Parse_Scan(FH,Adr);
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFDA SOS';
                 StringGrid1.Cells[1,Line]:='EndPos';
                 StringGrid1.Cells[2,Line]:=IntToHex(Adr,8);
               end;
        $d0ff,$d1ff,$d2ff,$d3ff,$d4ff,$d5ff,$d6ff,
        $d7ff: begin
                 StringGrid1.RowCount:=(Line+2)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:=IntToHex(Swap(FBuf.JFIFKen),4);
                 StringGrid1.Cells[1,Line]:='Typ';
                 StringGrid1.Cells[2,Line]:='Fortsetzung (Bilddaten)';
                 Adr:=Adr+2;
                 Parse_Scan(FH,Adr);
                 inc(Line);
                 StringGrid1.Cells[0,Line]:=IntToHex(Swap(FBuf.JFIFKen),4);
                 StringGrid1.Cells[1,Line]:='EndPos';
                 StringGrid1.Cells[2,Line]:=IntToHex(Adr,8);
               end;
        $c0ff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFC0 SOF0','Baseline DCT');
               end;
        $c1ff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFC1 SOF1','Ext. Sequent. DCT');
               end;
        $c2ff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFC2 SOF2','Progressive DCT');
               end;
        $c5ff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFC5 SOF5','Diff. Sequent. DCT');
               end;
        $c6ff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFC6 SOF6','Diff. Progr. DCT');
               end;
        $c9ff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFC9 SOF9','Ext. Sequent. DCT (AC)');
               end;
        $caff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFCA SOF10','Progressive DCT (AC)');
               end;
        $cdff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFCD SOF13','Diff. Sequent. DCT (AC)');
               end;
        $ceff: begin
                 inc(cntP);
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 Fill_CxInfo(Line,Adr,'FFCE SOF14','Diff. Progr. DCT (AC)');
               end;
        $e0ff: begin
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 if (FBuf.FFE0Ide[0]='J') and (FBuf.FFE0Ide[1]='F') and
                    (FBuf.FFE0Ide[2]='I') and (FBuf.FFE0Ide[3]='F') then begin
                   StringGrid1.RowCount:=(Line+5)+1;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='Typ';
                   StringGrid1.Cells[2,Line]:='JFIF-Header';
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='Lnge';
                   StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen));
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='JFIF-Vers';
                   StringGrid1.Cells[2,Line]:=IntToHex(Swap(FBuf.FFE0Ver),4);
                   case FBuf.FFE0AEi of
                     1 : HStr:=' (dpi)';
                     2 : HStr:=' (d/cm)';
                     else HStr:='';
                   end;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='Auflsung Typ';
                   StringGrid1.Cells[2,Line]:=IntToStr(FBuf.FFE0AEi)+HStr;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='Auflsung';
                   StringGrid1.Cells[2,Line]:=IntToStr(swap(FBuf.FFE0HAu))+'/'+
                                              IntToStr(swap(FBuf.FFE0VAu));
                 end
                 else if (FBuf.FFE0Ide[0]='J') and (FBuf.FFE0Ide[1]='F') and
                    (FBuf.FFE0Ide[2]='X') and (FBuf.FFE0Ide[3]='X') then begin
                   StringGrid1.RowCount:=(Line+2)+1;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='Typ';
                   StringGrid1.Cells[2,Line]:='JFIF-Erweiterung';
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                   StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                              IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
                 end
                 else begin
                   StringGrid1.RowCount:=(Line+2)+1;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='Kennung/Wert';
                   StringGrid1.Cells[2,Line]:=GetFBufStr;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE0 APP0';
                   StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                   StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                              IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
                 end;                             
               end;
        $e1ff: begin
                 Len:=FileRead(FH,FBuf.DummyB,min(Swap(FBuf.JFIFLen)-2,4000));
                 if (FBuf.FFE0Ide[0]='E') and (FBuf.FFE0Ide[1]='x') and
                    (FBuf.FFE0Ide[2]='i') and (FBuf.FFE0Ide[3]='f') then begin
                   StringGrid1.RowCount:=(Line+2)+1;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE1 APP1';
                   StringGrid1.Cells[1,Line]:='Typ';
                   StringGrid1.Cells[2,Line]:='EXIF-Daten';
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE1 APP1';
                   StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                   StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                              IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
                 end
                 else begin
                   StringGrid1.RowCount:=(Line+2)+1;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE1 APP1';
                   StringGrid1.Cells[1,Line]:='Kennung/Wert';
                   StringGrid1.Cells[2,Line]:=GetFBufStr;
                   inc(Line);
                   StringGrid1.Cells[0,Line]:='FFE1 APP1';
                   StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                   StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                              IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
                 end;
               end;
        $e2ff, $e3ff, $e4ff, $e5ff, $e6ff, $e7ff, $e8ff, $e9ff, $eaff, $ebff, $ecff, $edff, $eeff,
        $efff: begin
                 Len:=FileRead(FH,FBuf.DummyB,min(Swap(FBuf.JFIFLen)-2,4000));
                 StringGrid1.RowCount:=(Line+2)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:=IntToHex(Swap(FBuf.JFIFKen),4)+' APPx';
                 StringGrid1.Cells[1,Line]:='Kennung/Wert';
                 StringGrid1.Cells[2,Line]:=GetFBufStr;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:=IntToHex(Swap(FBuf.JFIFKen),4)+' APPx';
                 StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                 StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                            IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
               end;
        $c4ff: begin
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 StringGrid1.RowCount:=(Line+2)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFC4 DHT';
                 StringGrid1.Cells[1,Line]:='Typ';
                 StringGrid1.Cells[2,Line]:='Def. Huffman-Tabelle '+
                                            IntToStr(FBuf.FFC4CCp);
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFC4 DHT';
                 StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                 StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                            IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
               end;
        $ccff: begin
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 StringGrid1.RowCount:=(Line+2)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFCC DAC';
                 StringGrid1.Cells[1,Line]:='Typ';
                 StringGrid1.Cells[2,Line]:='Def. Arithm.-Kod. '+
                                            IntToStr(FBuf.FFC4CCp);
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFCC DAC';
                 StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                 StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                            IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
               end;
        $dbff: begin
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 StringGrid1.RowCount:=(Line+2)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFDB DQT';
                 StringGrid1.Cells[1,Line]:='Typ';
                 StringGrid1.Cells[2,Line]:='Def. Quant.-Tabelle '+
                                            IntToStr(FBuf.FFC4CCp);
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFDB DQT';
                 StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                 StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                            IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
               end;
        $ddff: begin
                 Len:=FileRead(FH,FBuf.DummyB,Swap(FBuf.JFIFLen)-2);
                 StringGrid1.RowCount:=(Line+2)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFDD DRI';
                 StringGrid1.Cells[1,Line]:='Typ';
                 StringGrid1.Cells[2,Line]:='Def. Restart Interval';
                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFDD DRI';
                 StringGrid1.Cells[1,Line]:='Intervall';
                 StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.FFDDInt));
{                 inc(Line);
                 StringGrid1.Cells[0,Line]:='FFDB DRI';
                 StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                 StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                            IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);}
               end;
          else begin
                 StringGrid1.RowCount:=(Line+1)+1;
                 inc(Line);
                 StringGrid1.Cells[0,Line]:=IntToHex(Swap(FBuf.JFIFKen),4);
                 StringGrid1.Cells[1,Line]:='Lnge/EndPos';
                 StringGrid1.Cells[2,Line]:=IntToStr(Swap(FBuf.JFIFLen))+'/'+
                                            IntToHex(Adr+Swap(FBuf.JFIFLen)+2,8);
               end;
      end;
      if (FBuf.JFIFKen=$d9ff) then begin
        Adr:=-2;
        Label12.Caption:='Ende durch Segment '+IntToHex(Swap(FBuf.JFIFKen),4)+
                         ' '+IntToStr(Line)+' Zeilen';
      end
      else if (FBuf.JFIFKen=$daff) or (FBuf.JFIFKen=$d0ff) or
              (FBuf.JFIFKen=$d1ff) or (FBuf.JFIFKen=$d2ff) or
              (FBuf.JFIFKen=$d3ff) or (FBuf.JFIFKen=$d4ff) or
              (FBuf.JFIFKen=$d5ff) or (FBuf.JFIFKen=$d6ff) or
              (FBuf.JFIFKen=$d7ff) then begin
        inc(cntD);
        Adr:=FileSeek(FH,Adr,0);
      end
      else begin
        Adr:=FileSeek(FH,Adr+Swap(FBuf.JFIFLen)+2,0);
        Label12.Caption:=InttoStr(Adr)+' ('+IntToHex(Adr,8)+')';
      end;
    until (Adr=FLen) or (Adr<0);
    Label12.Caption:=intToStr(cntP)+' Frames ('+intToStr(cntD)+' Scan-Blcke)';
  end;
end;

procedure TForm1.GetEXIFInfo(FH: Integer);
begin
  null;
end;

{************************************************************************}
{* Evt-Methode: Open1Click                                  (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      ffen des OpenDialogs (mit Filter) mit anschienden Auf- *}
{*              ruf der Methode GetFileInfo beim Bettigen des Men-    *}
{*              punktes (MainMenu/Datei/)Open                           *}
{* VCL-Event:   TForm1.Open1.OnClick                                    *}
{************************************************************************}

procedure TForm1.Open1Click(Sender: TObject);
begin
  OpenDialog1.Filter:='JFIF-JPEG-Bild (*.jpg,*.jpeg,*.jpe,*.jfif)|*.jpg;*.jpeg;*.jpe;*.jfif';
  if OpenDialog1.Execute then begin
    GetFileInfo(OpenDialog1.FileName);
  end
end;

{************************************************************************}
{* Pr.-Methode: WMDROPFILES                                 (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Auswertung der Message WM_DROPFILES mit anschienden Auf-*}
{*              ruf der Methode GetFileInfo falls Extension stimmt      *}
{* Win-Message: WM_DROPFILES;                                           *}
{************************************************************************}

procedure TForm1.WMDROPFILES(var Msg: TMessage);
var
  i, count, size: integer;
  dropfile: PChar;
  FileName: AnsiString;
begin
  inherited;
  count := DragQueryFile(Msg.WParam, $FFFFFFFF, dropfile, 255);
  for i := 0 to count - 1 do
  begin
    size := DragQueryFile(Msg.WParam, i, nil, 0) + 1;
    dropfile := StrAlloc(size);
    try
      DragQueryFile(Msg.WParam, i, dropfile, size);
      FileName:=StrPas(dropfile);
      if (AnsiUpperCase(ExtractFileExt(FileName))='.JPG') or
         (AnsiUpperCase(ExtractFileExt(FileName))='.JPEG') or
         (AnsiUpperCase(ExtractFileExt(FileName))='.JPE') or
         (AnsiUpperCase(ExtractFileExt(FileName))='.JFIF') then
        GetFileInfo(FileName);
    finally
      StrDispose(dropfile);
    end;
  end;
  DragFinish(Msg.WParam);
end;

{************************************************************************}
{* Evt-Methode: Close1Click                                 (0.1.0 B000)*}
{************************************************************************}
{* Inhalt:      Schlieen des Forms beim Bettigen des Menpunktes Close*}
{* VCL-Event:   TForm1.Close1.OnClick                                   *}
{************************************************************************}

procedure TForm1.Close1Click(Sender: TObject);
begin
  Close;
end;

{************************************************************************}
{* Evt-Methode: FormCreate                                  (0.1.0 B001)*}
{************************************************************************}
{* Inhalt:      Zustzliches Initialisieren vom File-Drag and Drop beim *}
{*              Erstellen des Forms                                     *}
{* VCL-Event:   TForm1.OnCreate                                         *}
{************************************************************************}

procedure TForm1.FormCreate(Sender: TObject);
begin
  DragAcceptFiles(Handle, true);
end;

end.
